# Copyright (c) 2019-2024 Broadcom. All Rights Reserved.
# Broadcom Confidential. The term "Broadcom" refers to Broadcom Inc.
# and/or its subsidiaries.

"""Constants
Constants that are used in the ImageManager.
"""

# Notification types.
try:
   from com.vmware.esx.settings_daemon_client import Notification
   INFO = Notification.Type.INFO
   WARNING = Notification.Type.WARNING
   ERROR = Notification.Type.ERROR
   NOTIFICATION_HAS_TYPE = True
except (ImportError, AttributeError):
   # settingsd binding is not present or does not have notification types.
   INFO = 'INFO'
   WARNING = 'WARNING'
   ERROR = 'ERROR'
   NOTIFICATION_HAS_TYPE = False

# Scanner Constants
# Status
COMPLIANT = "COMPLIANT"
NON_COMPLIANT = "NON_COMPLIANT"
STAGED = "STAGED"
NOT_STAGED = "NOT_STAGED"
INCOMPATIBLE = "INCOMPATIBLE"
UNAVAILABLE = "UNAVAILABLE"

# Impact types and messages.
IMPACT_NONE = "NO_IMPACT"
IMPACT_PARTIAL_MMODE = "PARTIAL_MAINTENANCE_MODE_REQUIRED"
IMPACT_MMODE = "MAINTENANCE_MODE_REQUIRED"
IMPACT_REBOOT = "REBOOT_REQUIRED"
IMPACT_UNKNOWN = "UNKNOWN"

# Impact from WCP/NSX components needs to be re-evaluated
NSX_COMPONENT = "nsx-lcp-bundle"
WCP_COMPONENT_PREFIX = "VMware-Spherelet-"

# Maintenance Mode types
FULL_MAINTENANCE_MODE = "fullMaintenanceMode"
PARTIAL_MAINTENANCE_MODE_WCP = "sphereletPartialMM"

# Component source, see SoftwareInfo.vmodl.
SOURCE_BASEIMAGE = "BASE_IMAGE"
SOURCE_ADDON = "ADD_ON"
SOURCE_HSP = "HARDWARE_SUPPORT_PACKAGE"
SOURCE_SOLUTION = "SOLUTION"
SOURCE_USER = "USER"
SOURCE_USER_REMOVED = "USER_REMOVED"

# Spec constants
DISP_VERSION = "display_version"
DISP_NAME = "display_name"
VERSION = "version"
VENDOR = "vendor"
BASE_IMG = "base_image"
BASEIMAGE_UI_NAME = "ESXi"
ADD_ON = "add_on"
COMPONENTS = "components"
HARDWARE_SUPPORT = "hardware_support"
SOLUTIONS = "solutions"
CONFIG_SCHEMA = 'config_schema'

# Notification ID and messages.
HOSTSCAN_PREFIX = "com.vmware.vcIntegrity.lifecycle.HostScan."
HOSTSEEDING_PREFIX = "com.vmware.vcIntegrity.lifecycle.HostSeeding."
HOST_PREFIX = "com.vmware.vcIntegrity.lifecycle.Host."
RESOLUTION_SUFFIX = ".Resolution"

UNAVAILABLE_ID = HOSTSCAN_PREFIX + "Unavailable"

# Host scan task notification
HOSTSCAN_STARTED = HOSTSCAN_PREFIX + "Started"
HOSTSCAN_FAILED = HOSTSCAN_PREFIX + "Failed"
HOSTSCAN_COMPLETED = HOSTSCAN_PREFIX + "Completed"

# Compliance status summary
_HOST_COMPLIANCE_PREFIX = HOSTSCAN_PREFIX + 'HostCompliance.'
HOST_COMPLIANCE_COMPLIANT_ID = _HOST_COMPLIANCE_PREFIX + 'Compliant'
HOST_COMPLIANCE_NONCOMPLIANT_ID = _HOST_COMPLIANCE_PREFIX + 'NonCompliant'
HOST_COMPLIANCE_QPACTION_NONCOMPLIANT_ID = \
   _HOST_COMPLIANCE_PREFIX + 'QuickPatchAction.NonCompliant'
HOST_COMPLIANCE_INCOMPATIBLE_ID = _HOST_COMPLIANCE_PREFIX + 'Incompatible'
HOST_COMPLIANCE_UNKNOWN_ID = _HOST_COMPLIANCE_PREFIX + 'Unknown'

_DPU_COMPLIANCE_PREFIX = HOSTSCAN_PREFIX + 'DPUCompliance.'
DPU_COMPLIANCE_COMPLIANT_ID = _DPU_COMPLIANCE_PREFIX + 'Compliant'
DPU_COMPLIANCE_NONCOMPLIANT_ID = _DPU_COMPLIANCE_PREFIX + 'NonCompliant'
DPU_COMPLIANCE_QPACTION_NONCOMPLIANT_ID = \
   _DPU_COMPLIANCE_PREFIX + 'QuickPatchAction.NonCompliant'
DPU_COMPLIANCE_INCOMPATIBLE_ID = _DPU_COMPLIANCE_PREFIX + 'Incompatible'
DPU_COMPLIANCE_UNKNOWN_ID = _DPU_COMPLIANCE_PREFIX + 'Unknown'

#  Component downgrades.
BASEIMAGE_DOWNGRADE_ID = HOSTSCAN_PREFIX + "BaseImageDowngrade"
BASEIMAGE_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "BaseImageComponentDowngrade"
ADDON_DOWNGRADE_ID = HOSTSCAN_PREFIX + "AddonDowngrade"
ADDON_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "AddonComponentDowngrade"
ADDON_COMP_BASEIMAGE_DOWNGRADE_ID = \
   HOSTSCAN_PREFIX + "AddonCompBaseImageDowngrade"
ADDON_COMP_HSP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "AddonCompHSPDowngrade"
COMPONENT_DOWNGRADE_ID = HOSTSCAN_PREFIX + "ComponentDowngrade"
HSP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPDowngrade"
HSP_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPComponentDowngrade"
HSP_COMP_BASEIMAGE_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPCompBaseImageDowngrade"
HSP_COMP_ADDON_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPCompAddonDowngrade"
SOLUTION_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "SolutionComponentDowngrade"
USER_COMP_BASEIMAGE_DOWNGRADE_ID = \
   HOSTSCAN_PREFIX + "UserCompBaseImageDowngrade"
USER_COMP_ADDON_DOWNGRADE_ID = HOSTSCAN_PREFIX + "UserCompAddonDowngrade"
USER_COMP_HSP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "UserCompHSPDowngrade"
USER_COMP_SOLUTION_DOWNGRADE_ID = HOSTSCAN_PREFIX + "UserCompSolutionDowngrade"
VIB_DOWNGRADE_ID = HOSTSCAN_PREFIX + "VibDowngrade"

ADDON_REMOVAL_ID = HOSTSCAN_PREFIX + "AddonRemoval"
COMPONENT_REMOVAL_ID = HOSTSCAN_PREFIX + "ComponentRemoval"
HSP_REMOVAL_ID = HOSTSCAN_PREFIX + "HSPRemoval"
SOLUTIONCOMPONENT_REMOVAL_ID = HOSTSCAN_PREFIX + "SolutionComponentRemoval"
VIB_REMOVAL_ID = HOSTSCAN_PREFIX + "VibRemoval"

SOLUTION_DISABLE_ID = HOSTSCAN_PREFIX + "SolutionDisable"
SOLUTION_ENABLE_ID = HOSTSCAN_PREFIX + "SolutionEnable"

BASEIMAGE_COMPONENT_REMOVED_ID = HOSTSCAN_PREFIX + "BaseImageComponentRemoved"
ADDON_COMPONENT_REMOVED_ID = HOSTSCAN_PREFIX + "AddonComponentRemoved"
HSP_COMPONENT_REMOVED_ID = HOSTSCAN_PREFIX + "HSPComponentRemoved"
USER_REMOVED_COMPONENT_ID = HOSTSCAN_PREFIX + "UserRemovedComponent"
USER_REMOVED_COMP_ADDON_ID = HOSTSCAN_PREFIX + "UserRemovedAddOnComponent"
USER_REMOVED_COMP_BI_ID = HOSTSCAN_PREFIX + "UserRemovedBaseImageComponent"
USER_REMOVED_COMP_HSP_ID = HOSTSCAN_PREFIX + "UserRemovedHSPComponent"
USER_REMOVED_COMP_USER_ID = HOSTSCAN_PREFIX + "UserRemovedUserComponent"

MAINTMODE_IMPACT_ID = HOSTSCAN_PREFIX + "MaintenanceModeImpact"
REBOOT_IMPACT_ID = HOSTSCAN_PREFIX + "RebootImpact"
PARTIAL_MAINTMODE_IMPACT_ID = HOSTSCAN_PREFIX + "PartialMaintenanceModeImpact"
PENDING_REBOOT_ID = HOSTSCAN_PREFIX + "PendingReboot"

INSUFFICIENT_MEMORY_ID = HOSTSCAN_PREFIX + "InsufficientMemory"

UNSUPPORTED_CPU_ID = HOSTSCAN_PREFIX + "UnsupportedCpu"
UNSUPPORTED_CPU_WARN_ID = HOSTSCAN_PREFIX + "UnsupportedCpuWarn"
UNSUPPORTED_CPU_OVERRIDE_ID = HOSTSCAN_PREFIX + "UnsupportedCpuOverride"

NOT_ENOUGH_CPU_CORE_ID = HOSTSCAN_PREFIX + "NotEnoughCpuCore"
UNSUPPORTED_PCI_DEVICE = HOSTSCAN_PREFIX + "UnsupportedPciDevice"
# Unsupported Model or Vendor
UNSUPPORTED_VENDOR_MODEL_ID = HOSTSCAN_PREFIX + "Unsupported%s"
NO_NIC_DRIVER_ID = HOSTSCAN_PREFIX + "NoNicDriver"
NO_BOOTDEVICE_DRIVER_ID = HOSTSCAN_PREFIX + "NoBootDeviceDriver"
INSUFFICIENT_BOOTBANK_SPACE_ID = HOSTSCAN_PREFIX + "InsufficientBootBankSpace"
INSUFFICIENT_LOCKER_SPACE_ID = HOSTSCAN_PREFIX + "InsufficientLockerSpace"
DISK_TOO_SMALL_ID = HOSTSCAN_PREFIX + "DiskTooSmall"
DISK_TOO_SMALL_WARN_ID = HOSTSCAN_PREFIX + "DiskTooSmallWarn"
HOST_ACCEPTANCE_ID = HOSTSCAN_PREFIX + "HostAcceptance"
VSAN_V1_DISK_ID = HOSTSCAN_PREFIX + "VsanV1Disk"
UNSUPPORTED_TPM_ID = HOSTSCAN_PREFIX + "UnsupportedTPMVersion"
UNSUPPORTED_BIOS_FIRMWARE_ID = HOSTSCAN_PREFIX + "UnsupportedBiosFirmwareType"
MISSING_GUNZIP_CHECKSUM_VIB_ID = HOSTSCAN_PREFIX + "MissingGunzipChecksumVibs"
INCOMPAT_SPHERELET_VER_ID = HOSTSCAN_PREFIX + "IncompatibleSphereletVersion"
UNSUPPORTED_SHA1_CERT = HOSTSCAN_PREFIX + "UnsupportedSHA1Cert"

INSUFFICIENT_ENTROPY_CACHE = HOSTSCAN_PREFIX + "InsufficientEntropyCache"
VFAT_CORRUPTION = HOSTSCAN_PREFIX + "VfatCorruption"

# Live Patch specific.
_QP_PREFIX = HOSTSCAN_PREFIX + 'QuickPatch.'
QP_INFO_ID = _QP_PREFIX + 'Eligible'
QP_TPM_UNSUPPORTED_ID = _QP_PREFIX + 'TpmUnsupported'
QP_DPU_UNSUPPORTED_ID = _QP_PREFIX + 'DpuUnsupported'
QP_LIVE_COMP_UNSUPPORTED_ID = _QP_PREFIX + 'LiveCompUnsupported'
QP_LIVE_ORPHAN_VIB_UNSUPPORTED_ID = _QP_PREFIX + 'LiveOrphanVibUnsupported'
QP_LIVE_SOLUTION_UNSUPPORTED_ID = _QP_PREFIX + 'LiveSolutionUnsupported'
QP_SCANSCRIPT_FAILURE_ID = _QP_PREFIX + 'ScanScriptFailure'
QP_INCOMPLETE_REBOOT_ID = _QP_PREFIX + 'IncompleteRemediation.RebootRequired'

_QP_HOST_PREFIX = HOST_PREFIX + 'QuickPatch.'
QP_SCRIPT_STARTING_ID = _QP_HOST_PREFIX + 'ScriptStarting'
QP_SCRIPT_SUCCEEDED_ID = _QP_HOST_PREFIX + 'ScriptSucceeded'
QP_SCRIPT_FAILED_ID = _QP_HOST_PREFIX + 'ScriptFailed'
QP_SCRIPT_INFO_ID = _QP_HOST_PREFIX + 'ScriptInfo'
QP_SCRIPT_WARNING_ID = _QP_HOST_PREFIX + 'ScriptWarning'
QP_SCRIPT_ERROR_ID = _QP_HOST_PREFIX + 'ScriptError'

QP_SCRIPTMSG_TYPE_TO_WRAPPER_ID = {
   INFO: QP_SCRIPT_INFO_ID,
   WARNING: QP_SCRIPT_WARNING_ID,
   ERROR: QP_SCRIPT_ERROR_ID,
}

# Host seeding related
COMPONENTS_DOWNGRADED = HOSTSEEDING_PREFIX + "ComponentsDowngraded"
COMPONENTS_REMOVED = HOSTSEEDING_PREFIX + "ComponentsRemoved"
ORPHAN_VIB = HOSTSEEDING_PREFIX + "OrphanVib"
SEEDING_PENDING_REBOOT_ID = HOSTSEEDING_PREFIX + "PendingReboot"
UNSUPPORTED_STATELESS_HOST_ID = HOSTSEEDING_PREFIX + "UnsupportedStatelessHost"

# QuickBoot compatibility check
DRIVER_MAP_DIRS = set([
   "etc/vmware/default.map.d",
   "etc/vmware/fallback.map.d",
   "etc/vmware/ihv.map.d",
   "etc/vmware/oem.map.d",
])

_QUICKBOOT_PREFIX = HOSTSCAN_PREFIX + "QuickBoot."
QUICKBOOT_UNSUPPORTED_ID = _QUICKBOOT_PREFIX + "Unsupported"
QUICKBOOT_UNAVAILABLE_DRIVER_DG_ID = \
   _QUICKBOOT_PREFIX + "Unavailable.DriverDowngrade"
QUICKBOOT_SUPPORTED_ID = _QUICKBOOT_PREFIX + "Supported"

# QuickBoot and SuspendToMemory compatibility check
_QUICKBOOT_AND_SUSPEND_TO_MEMORY_PREFIX = \
   HOSTSCAN_PREFIX + "QuickBootAndSuspendToMemory."
QUICKBOOT_AND_SUSPEND_TO_MEMORY_SUPPORTED_ID = \
   _QUICKBOOT_AND_SUSPEND_TO_MEMORY_PREFIX + "Supported"

# Error message IDs, default messages and arguments.
ESXIMAGE_PREFIX = "com.vmware.vcIntegrity.lifecycle.EsxImage."

#  Usually the name of the exeption will be the key in APPLY_EXCEPTION_MSG_ARG,
#  here aliases are included to convert an exception.
ESXIMAGE_ERROR_ALIAS = {
   "ComponentConfigDowngrade": "ComponentDowngradeError",
   # Profile validation contains a few other checks, but generally it is
   # correct to say the components have validation issues.
   "ProfileValidationError": "ComponentValidationError",
   "DependencyError": "ComponentValidationError",
}

NOTIFICATION_MSG = {
   # Compliance status details.
   HOST_COMPLIANCE_COMPLIANT_ID: "Host is compliant with the image.",
   HOST_COMPLIANCE_NONCOMPLIANT_ID: "Host is out of compliance with the image.",
   HOST_COMPLIANCE_QPACTION_NONCOMPLIANT_ID: \
      "Host is out of compliance due to pending remediation actions.",
   HOST_COMPLIANCE_INCOMPATIBLE_ID: "Host is not compatible with the image.",
   HOST_COMPLIANCE_UNKNOWN_ID: "Host status is unknown.",

   DPU_COMPLIANCE_COMPLIANT_ID: "DPU is compliant with the image.",
   DPU_COMPLIANCE_NONCOMPLIANT_ID: "DPU is out of compliance with the image.",
   DPU_COMPLIANCE_QPACTION_NONCOMPLIANT_ID: \
      "DPU is out of compliance due to pending remediation actions.",
   DPU_COMPLIANCE_INCOMPATIBLE_ID: "DPU is not compatible with the image.",
   DPU_COMPLIANCE_UNKNOWN_ID: "DPU status is unknown.",

   UNAVAILABLE_ID: "Host compliance check failed with an unknown error.",
   BASEIMAGE_DOWNGRADE_ID: "ESXi version {1} in the image is a downgrade of " \
      "the installed version {2}.",
   BASEIMAGE_COMP_DOWNGRADE_ID: "Downgrades of ESXi Components {1} are not " \
      "supported.",
   ADDON_DOWNGRADE_ID: "Version {1} of vendor Addon {2} in the image is a " \
      "downgrade of the installed version {3}.",
   ADDON_COMP_DOWNGRADE_ID: "Downgrades of Addon Components {1} in the " \
      "desired Addon {2} are not supported.",
   ADDON_COMP_BASEIMAGE_DOWNGRADE_ID: "Downgrades of Addon Components {1} in " \
      "the desired ESXi version are not supported.",
   ADDON_COMP_HSP_DOWNGRADE_ID: "Downgrades of Addon Components {1} in the " \
      "desired Hardware Support Package {2} are not supported.",
   HSP_DOWNGRADE_ID: "Version {1} of Hardware Support Package {2} in the " \
      "image is a downgrade of the installed version {3}.",
   HSP_COMP_DOWNGRADE_ID: "Downgrades of Hardware Support Package Components " \
      "{1} in the desired Hardware Support Package {2} are not supported.",
   HSP_COMP_BASEIMAGE_DOWNGRADE_ID: "Downgrades of Hardware Support Package " \
      "Components {1} in the desired ESXi version are not supported.",
   HSP_COMP_ADDON_DOWNGRADE_ID: "Downgrades of Hardware Support Package " \
      "Components {1} in the desired Addon {2} are not supported.",
   COMPONENT_DOWNGRADE_ID: "Version {1} of the manually added Component {2} " \
      "is an unsupported downgrade from version {3} on the host.",
   SOLUTION_COMP_DOWNGRADE_ID: "Downgrades of Components {1} in Solution {2} " \
      "are not supported.",
   USER_COMP_BASEIMAGE_DOWNGRADE_ID: "Downgrades of manually added " \
      "Components {1} in the desired ESXi version are not supported.",
   USER_COMP_ADDON_DOWNGRADE_ID: "Downgrades of manually added Components " \
      "{1} in the desired Addon {2} are not supported.",
   USER_COMP_HSP_DOWNGRADE_ID: "Downgrades of manually added Components {1} " \
      "in the desired Hardware Support Package {2} are not supported.",
   USER_COMP_SOLUTION_DOWNGRADE_ID: "Downgrades of manually added Components " \
      "{1} in the desired Solution {2} are not supported.",
   VIB_DOWNGRADE_ID: "Downgrades of host VIBs {1} in the image are not " \
      "supported.",
   ADDON_REMOVAL_ID: "Vendor Addon {1} with version {2} is removed in the " \
      "image. It will be removed from the host during remediation.",
   HSP_REMOVAL_ID: "Hardware Support Package {1} with version {2} is " \
      "removed in the image. It will be removed from the host during " \
      "remediation.",
   COMPONENT_REMOVAL_ID: "Components {1} are removed in the image. They will " \
      "be removed from the host during remediation.",
   SOLUTIONCOMPONENT_REMOVAL_ID: "Components {1} in Solution {2} are removed " \
      "in the image. They will be removed from the host during remediation.",
   VIB_REMOVAL_ID: "The following VIBs on the host are missing from the " \
      "image and will be removed from the host during remediation: {1}.",

   SOLUTION_DISABLE_ID: "Solution Components of disabled Solutions {1} will " \
                        "be removed from this host during remediation.",
   SOLUTION_ENABLE_ID: "Remediating this host will enable these Solutions: " \
                       "{1}.",

   BASEIMAGE_COMPONENT_REMOVED_ID: "ESXi Components {1} are either missing " \
      "or downgraded on the host. Appropriate action will be taken during " \
      "remediation.",
   ADDON_COMPONENT_REMOVED_ID: "Vendor Addon Components {1} are either " \
      "missing or downgraded on the host. Appropriate action will be taken " \
      "during remediation.",
   HSP_COMPONENT_REMOVED_ID: "Hardware Support Package Components {1} are " \
      "either missing or downgraded on the host. The correct versions of " \
      "the Components will be installed during remediation.",
   USER_REMOVED_COMPONENT_ID: "User has removed the component {1} in the " \
      "image. During remediation, the component will not be installed and " \
      "will be removed from the host if present.",
   USER_REMOVED_COMP_ADDON_ID: "User has removed the component {1} from the " \
      "Addon in the image. During remediation, the component will not be " \
      "installed and will be removed from the host if present.",
   USER_REMOVED_COMP_BI_ID: "User has removed the component {1} from the " \
      "Base Image in the image. During remediation, the component will not " \
      "be installed and will be removed from the host if present.",
   USER_REMOVED_COMP_HSP_ID: "User has removed the component {1} from the " \
      "Hardware Support Package in the image. During remediation, the " \
      "component will not be installed and will be removed from the host if " \
      "present.",
   USER_REMOVED_COMP_USER_ID: "User has removed the component {1} from the " \
      "user components in the image. During remediation, the component will " \
      "not be installed and will be removed from the host if present.",
   MAINTMODE_IMPACT_ID: "The host will be placed in maintenance mode before " \
      "remediation.",
   REBOOT_IMPACT_ID: "The host will be rebooted during remediation.",
   PARTIAL_MAINTMODE_IMPACT_ID: "The host will be placed in partial " \
                                "maintenance mode before remediation.",
   PENDING_REBOOT_ID: "The host is pending a reboot since a previous " \
                      "software change. The pending change is not reflected " \
                      "in host's current state in the compliance results " \
                      "below. Reboot the host to see accurate compliance " \
                      "results.",

   # Live Patch specific.
   QP_INFO_ID: "The host is eligible for Live Patch. Remediation " \
      "can be performed without rebooting or going into maintenance mode.",
   QP_TPM_UNSUPPORTED_ID: "The image is eligible for Live Patch. However, " \
      "Live Patch is currently unsupported on a host with TPM enabled.",
   QP_DPU_UNSUPPORTED_ID: "The image is eligible for Live Patch. However, " \
      "Live Patch is currently unsupported on a host with managed DPU(s).",
   QP_LIVE_COMP_UNSUPPORTED_ID: "Live Patch cannot be performed together " \
      "with component(s) {1} that will be remediated live.",
   QP_LIVE_ORPHAN_VIB_UNSUPPORTED_ID: "Live Patch cannot be performed " \
      "together with live removal of VIB(s) {1}.",
   QP_LIVE_SOLUTION_UNSUPPORTED_ID: "Live Patch cannot be performed " \
      "together with solution(s) {1} that will be enabled or disabled live.",
   QP_SCANSCRIPT_FAILURE_ID: "Live Patch check compliance script(s) '{1}' " \
      "failed to execute or returned invalid output.",
   QP_INCOMPLETE_REBOOT_ID: "One or more Live Patch action is pending after " \
      "the previous remediation. As the Live Patch remediation policy is " \
      "disabled, a reboot is required for the host to become compliant.",

   QP_SCRIPT_STARTING_ID: "Starting to execute Live Patch {1} script '{2}'.",
   QP_SCRIPT_SUCCEEDED_ID: "Live Patch {1} script '{2}' succeeded.",
   QP_SCRIPT_FAILED_ID: "Live Patch {1} script '{2}' failed.",

   # Wrappers for Live Patch script notifications, unlocalized message will be
   # filled in as argument {3}.
   QP_SCRIPT_INFO_ID: "Live Patch {1} script '{2}' returned a message: {3}",
   QP_SCRIPT_WARNING_ID: "Live Patch {1} script '{2}' returned a warning: {3}",
   QP_SCRIPT_ERROR_ID: "Live Patch {1} script '{2}' returned an error: {3}",

   INSUFFICIENT_MEMORY_ID: "The host has {1} MB of memory, a minimum of {2} " \
      "MB is required by the image.",
   UNSUPPORTED_CPU_ID: "The CPU on this host is not supported by the image. " \
      "Please refer to KB 82794 for more details.",
   UNSUPPORTED_CPU_WARN_ID: "The CPU on this host may not be supported in " \
      "future ESXi releases. Please plan accordingly. " \
      "Please refer to KB 82794 for more details.",
   UNSUPPORTED_CPU_OVERRIDE_ID: "The CPU on this host is not supported by " \
      "ESXi. You can override and force remediate, but it is neither " \
      "officially supported nor recommended. Please refer to KB 82794 for " \
      "more details.",
   UNSUPPORTED_BIOS_FIRMWARE_ID: "Legacy boot detected. ESXi servers running " \
      "legacy BIOS are encouraged to move to UEFI. Please refer to KB 84233 " \
      "for more details.",
   NOT_ENOUGH_CPU_CORE_ID: "The CPU on the host has {1} cores, a minimum of " \
      "{2} cores are required by the image.",
   UNSUPPORTED_PCI_DEVICE: "Unsupported PCI devices {1} found on the host.",
   UNSUPPORTED_VENDOR_MODEL_ID % 'Vendor': "Host hardware vendor {1} is not " \
      "supported by the image.",
   UNSUPPORTED_VENDOR_MODEL_ID % 'Model': "Host hardware model {1} is not " \
      "supported by the image.",
   NO_NIC_DRIVER_ID: "The management NIC of the host lacks a driver in the " \
      "image.",
   NO_BOOTDEVICE_DRIVER_ID: "The boot device of the host lacks a driver in " \
      "the image.",
   INSUFFICIENT_BOOTBANK_SPACE_ID: "The bootbank partition on the host has a " \
      "capacity of {1} MB, the image requires {2} MB.",
   INSUFFICIENT_LOCKER_SPACE_ID: "The locker partition on the host has {1} " \
      "MB free space available, the image requires {2} MB.",
   DISK_TOO_SMALL_ID: "The boot device of the host has a capacity of {1} " \
      "MiB, a minimum of {2} MiB is required by the image.",
   DISK_TOO_SMALL_WARN_ID: "The boot device of the host has a capacity of " \
      "{1} MiB. Install or upgrade to future ESXi releases will not be " \
      "supported on this boot device. Please switch to a boot device that " \
      "is at least {2} MiB in size. Refer to KB-96511 for further details.",
   HOST_ACCEPTANCE_ID: "Acceptance level of the image is not " \
      "compatible with the host's configured setting.",
   MISSING_GUNZIP_CHECKSUM_VIB_ID: "These VIB(s) on the host do not have the" \
      " required sha-256 gunzip checksum for their payloads:\n{1}\nThis will" \
      " prevent VIB security verification and secure boot from functioning" \
      " properly. Please remove these VIB(s) or please check with your"\
      " vendor for a replacement of these VIB(s)",
   INSUFFICIENT_ENTROPY_CACHE: "Storage entropy cache is not full. A full " \
      "entropy cache is required for upgrade. Refer to KB 89854 for steps on " \
      "how to refill the cache.",
   QUICKBOOT_UNSUPPORTED_ID: "Quick Boot is not supported on the host.",
   QUICKBOOT_UNAVAILABLE_DRIVER_DG_ID: "Quick Boot is not available due to " \
      "downgrade of driver(s): {1}.",
   QUICKBOOT_SUPPORTED_ID: "Quick Boot is supported on the host.",
   QUICKBOOT_AND_SUSPEND_TO_MEMORY_SUPPORTED_ID: "Quick Boot and "
      "Suspend To Memory are supported on the host.",
   VSAN_V1_DISK_ID: "vSAN v1 disks are not supported for upgrade.",
   UNSUPPORTED_TPM_ID: "TPM 1.2 device detected. Support for TPM version 1.2 " \
      "is discontinued. Installation may proceed, but may cause the system " \
      "to behave unexpectedly.",
   INCOMPAT_SPHERELET_VER_ID: "Spherelet version {1} is not compatible with " \
      "ESXi 8.0 and later version. Please upgrade your WCP cluster to " \
      "install a compatible Spherelet version, or remove Spherelet if the " \
      "host is not in a WCP cluster.",
   UNSUPPORTED_SHA1_CERT: "SHA-1 signature found in host certificate {1}. " \
      "Support for certificates with weak signature algorithm SHA-1 has been " \
      "removed in ESXi 8.0. To proceed with upgrade, replace it with a SHA-2 " \
      "signature based certificate. Refer to the release notes and KB 89424 " \
      "for more details.",
   COMPONENTS_DOWNGRADED: "The following Components have been downgraded on " \
      "the host: {1}",
   COMPONENTS_REMOVED: "The following Components have been removed on the " \
      "host: {1}",
   ORPHAN_VIB: "The following standalone VIB(s) are present on the host " \
      "and will not be extracted: {1}",
   SEEDING_PENDING_REBOOT_ID: "The host is pending a reboot since a " \
      "previous software change. The pending change is not reflected " \
      "in the software info and will not be extracted into the depot.",
   UNSUPPORTED_STATELESS_HOST_ID: "Getting software info and depot " \
      "extraction are not supported on a stateless host.",
   HOSTSCAN_STARTED: "Host image compliance check started.",
   HOSTSCAN_FAILED: "Host image compliance check failed.",
   HOSTSCAN_COMPLETED: "Host image compliance check successfully completed.",
   VFAT_CORRUPTION: "A problem with one or more vFAT bootbank partitions " \
                    "was detected. Please refer to KB 91136 and run dosfsck " \
                    "on bootbank partitions.",
   }

#  Map from name of exception to default message and arguments to extract from
#  the exception.
ESXIMAGE_ERROR_MSG_ARG = {
   "AcceptanceConfigError": ("A VIB package cannot be installed due to the "
                             "current host acceptance level configuration.",
                             []),
   "AcceptanceGetError": ("Failed to get the current host acceptance level "
                          "configuration.",
                          []),
   "AddonBaseImageMismatchError": ("Addon {1} is incompatible with base "
                                   "image version {2}.",
                                   ["addonId", "baseImageVersion"]),
   "AddonNotFound": ("Addon {1} with version {2} cannot be found in depot.",
                     ["name", "version"]),
   "AddonValidationError": ("Failed to validate metadata or components of " \
                            "addon {1}.",
                            ['addonID']),
   "ApplyHARebootRequiredError": ("Failed to apply HA due to the host "
                                  "software configuration requires a reboot. "
                                  "Reboot the host and try again.", []),
   "BaseImageNotFound": ("The Base Image with version {1} cannot be found in "
                         "depot.",
                         ["version"]),
   "BaseimageValidationError": ("Failed to validate metadata or components " \
                                "of the base image.",
                                []),
   "ComponentDowngradeError": ("The operation cannot continue due to downgrade "
                               "of the following Components: {1}.",
                               ["components"]),
   "ComponentFormatError": ("An error occurred while parsing or merging "
                            "components.",
                            []),
   "ComponentNotFoundError": ("Component {1} cannot be found in depot.",
                              ["name"]),
   "ComponentUnmountError": ("Failed to remove Component {1}, files may "
      "still be in use.",
      ["component"]),
   "ComponentValidationError": ("There are Compoment relation and content "
                                "problems with the desired state.",
                                []),
   "ConfigComponentDowngradeError": ("Component '{1}' cannot be downgraded due "
      "to potential configuration incompatibility.",
      ["components"]),
   "DatabaseIOError": ("Cannot read or write the image database on the host.",
                       []),
   "DatabaseFormatError": ("The image database on the host is invalid or "
                           "corrupted.",
                           []),
   "DepotConnectError": ("An error occurred while connecting to depot.",
                         []),
   "DpuOperationError": ("The operation on DPU failed.", []),
   "ESXioComponentDowngradeError": ("The operation cannot continue due to "
      "downgrade of the following ESXio Components: {1}.", ["components"]),
   "HardwareSupportPackageNotFound": (
      "Manifest with Hardware Support Manager {1} and Hardware Support "
      "Package {2} version {3} cannot be found in depot.",
      ["manager", "package", "version"]),
   "IncompatibleSolutionCompsError": ("No compatible combination of Components "
      "{1} is found in Solution {2}.",
      ["compNames", "solutionName"]),
   "IncompatibleSolutionComponentError": ("No compatible Component {1} is "
      "found in Solution {2}, candidate versions are: {3}.",
      ["name", "solutionName", "compVersions"]),
   "InstallationError": ("An internal error occurred "
                         "while staging/remediating the host.",
                         []),
   "LiveInstallationError": ("An error occurred while performing live system "
                             "changes during staging/remediation.",
                             []),
   "DataStorageNotFound": ("No OSData storage partition is available to "
      "extract depot. Configure persistent storage for the host and retry.",
      []),
   "LockingError": ("Remediation cannot begin because another process is "
                    "updating the host.",
                    []),
   "MaintenanceModeError": ("Maintenance mode is either not enabled or cannot "
                            "be determined on the host.",
                            []),
   "MemoryReserveError": (
      "The host is out of resource: failed to reserve {1} MB of memory during "
      "remediation.",
      ["reserveSize"]),
   "ManifestBaseImageMismatchError": (
      "Manifest with Hardware Support Manager {1} and Hardware Support "
      "Package {2} version {3} does not support Base Image version {4}.",
      ["manager", "package", "version", "baseImageVersion"]),
   "ManifestConflictError":(
      "Hardware Support Packages with ID {1} and {2} have overlapping "
      "Base Image support. Use a different package that does not overlap "
      "with package {1}.",
      ["manifestID", "newManifestID"]),
   "MetadataDownloadError": ("An error occurred while downloading depot "
                             "metadata from {1}.",
                             ['url']),
   "MetadataFormatError": ("The format of metadata in {1} is invalid.",
                           ['filename']),
   "MetadataIOError": ("An error occured while reading metadata.zip.",
                       []),
   "MissingComponentError": ("Cannot find metadata for a Component.",
                             []),
   "MissingVibError": ("Cannot find metadata for VIB(s) {1} in database or "
                       "depot.",
                       ["vibIDs"]),
   "MultipleManifestError": (
      "The current version of ESXi does not support multiple Hardware Support "
      "Packages.",
      []),
   "PartialVibDownloadError": ("Failed to download Live Patch VIB from '{1}'.",
      ['url']),
   "QuickPatchIncompleteError": ("The Live Patch image was remediated. "
      "However, an error occurred while performing post-remediation actions.",
      []),
   "QuickPatchIncompleteScriptError": ("The Live Patch image was remediated. "
      "However, an error occurred while executing post-remediation Live Patch "
      "script(s) '{1}'.", ['scriptNames']),
   "QuickPatchInstallationError": ("An internal error occurred while "
      "performing Live Patch staging or remediation.", []),
   "QuickPatchScriptError": ("Failed to execute Live Patch script '{1}'.",
      ['scriptNames']),
   "RemovedComponentNotFound": ("Component {1} cannot be removed because it is "
      "not found in the image.",
      ['name']),
   "ReleaseUnitConflictError": ("There are unequal Base Images, Addons or "
                                "Solutions metadata that share the "
                                "same ID {1}.",
                                ['releaseID']),
   "ReleaseUnitSchemaVersionError": ("The schema version {1} of {2}: {3} is "
      "incompatible with the current schema version {4}. Please upgrade this "
      "product to a version that is compatible with the schema version.",
      ["schemaVer", "releaseUnitType", "releaseID", "compatibleSchemaVer"]),
   "ReservedVibExtractError": ("Failed to extract depot from the host: no "
                               "stored copy available for inactive VIB {1}.",
                               ['vib']),
   "ResourceError": ("No space available to stage contents", []),
   "ResourcePoolError": ("Resource pool '{1}' cannot be created or removed. "
      "The host may be low on memory or the resource pool may still be used "
      "by a process.", ['rpName']),
   "ResourcePoolFileNotFound": ("VIB {1} does not have a resource pool "
      "definition file. Make sure a valid VIB is used for the operation.",
      ['vibID']),
   "ResourcePoolFileInvalid": ("Resource pool definition file '{1}' is "
      "invalid. Make sure a valid VIB is used for the operation.",
      ['rpFile']),
   "SecurityPolicyError": ("Security policies in directory '{1}' cannot be "
      "loaded. Make sure the path is valid and the directory has not been "
      "tampered with.", ['policyDir']),
   "SecurityPolicyNotFound": ("VIB {1} does not have a security policy "
      "directory defined. Make sure a valid VIB is used for the operation.",
      ['vibID']),
   "ServiceDisableError": ("Failed to remove Component {1}: an error "
      "occurred while disabling service {2}.",
      ["component", "service"]),
   "ServiceEnableError": ("Failed to apply Component {1}: an error "
      "occurred while enabling service {2}.",
      ["component", "service"]),
   "SoftwareInfoExtractError": (
      "An error occurred while extracting image info on the host.",
      ['msg']),
   "SoftwareSpecFormatError": ("Corrputed or invalid software specification "
                               "format. Missing attribute: {1}.",
                               ['field']),
   "SolutionComponentNotFound": ("Component(s) {1} within software Solution {2}"
                                 " cannot be found in depot.",
                                 ["compNames", "solutionName"]),
   "SolutionNotFound": ("Software Solution {1} with version {2} cannot be "
                        "found in depot.",
                        ["name", "version"]),
   # vLCM supports only stateful hosts, StatelessError means
   # there is an issue with /bootbank or /altbootbank.
   "StatelessError": ("The bootbank partition is not accessible " \
                      "or contains an invalid image database. " \
                      "Check /bootbank, /altbootbank paths "\
                      "are valid and accessible.",
                      []),
   "UnsupportedRemovedComponent": ("Removal of component {1} is not supported. "
      "Only certain components in the Base Image and components that are "
      "only found in the OEM Addon can be removed.",
      ['name']),
   "VibCertStoreError": ("Failed to import certificates or revocation lists "
                         "while verifying VIB signature.",
                         []),
   "VibChecksumError": ("Recreation of the VIB {1} failed due to checksum "
                        "mismatch. VIB may have been created using older "
                        "tools. Try a different host or update the component "
                        "that contains this VIB.",
                        ['vib']),
   "VibDownloadError": ("An error occurred while downloading VIB from URL {1}.",
                        ['url']),
   "VibFormatError": ("The format of VIB {1} is invalid.",
                      ['filename']),
   "VibHwPlatformError": ("Hardware vendor and model requirements of "
                          "VIBs cannot be satisfied by this host. Use VIBs "
                          "that match hardware vendor and model of host.",
                          []),
   "VibRecreateError": ("Failed to extract depot from the host: an error "
                        "occurred while re-creating VIB {1}.",
                        ['vib']),
   "VibSigDigestError": ("Signature of VIB {1} cannot be verified against its "
                         "contents.",
                         ['vib']),
   "VibSigFormatError": ("VIB {1} contains an invalid or corrupted signature.",
                         ['vib']),
   "VibSigInvalidError": ("VIB {1} contains an invalid or corrupted signature.",
                          ['vib']),
   "VibSigMissingError": ("VIB {1} does not have a signature and thus cannot "
                          "be verified.",
                          ['vib']),
   "VibXmlSchemaLoadError": ("Failed to load XML schema for VIB validation. "
                             "Check if XML schema file {1} is present on the "
                             "host.",
                             ['filename']),
   # Catch-all unknown error message that need to be used when the type is not
   # listed above.
   "UnknownError": ("An unknown error occurred while performing the operation.",
                    []),
   }

# Resolution messages, indexed by notification/message IDs.
#   When an ID does not have a message here, the resolution structure of the
#   notification will automatically have empty ID and message.
_COMMON_ADD_USERCOMP_RES = "Add the Components of matching or " \
   "higher versions to the image."
_COMMON_UPDATE_SOLUTION_COMP_RES = "Check the Solution's compatibility " \
   "matrix to enable higher version Component(s), or disable the Solution."
RESOLUTION_MSG = {
   UNAVAILABLE_ID: "Retry check compliance at a later time.",
   BASEIMAGE_DOWNGRADE_ID: "Replace the ESXi version in the image with one " \
      "of the same or higher version.",
   BASEIMAGE_COMP_DOWNGRADE_ID: "Select a Base Image that does not downgrade " \
      "an installed Base Image Component.",
   ADDON_DOWNGRADE_ID: "Replace the vendor Addon in the image with one of " \
      "the same or higher version.",
   ADDON_COMP_DOWNGRADE_ID: "Select an vendor Addon that does not downgrade " \
      "an installed Addon Component.",
   ADDON_COMP_BASEIMAGE_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   ADDON_COMP_HSP_DOWNGRADE_ID: "Select a Hardware Support Package that " \
      "contains Components of matching or higher versions.",
   HSP_DOWNGRADE_ID: "Select a matching or higher version of Hardware " \
                     "Support Package.",
   HSP_COMP_DOWNGRADE_ID: "Select a Hardware Support Package that does not " \
      "downgrade an installed Hardware Support Package Component.",
   HSP_COMP_BASEIMAGE_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   HSP_COMP_ADDON_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   COMPONENT_DOWNGRADE_ID: "Replace the Component in the image with one of " \
      "the same or higher version.",
   SOLUTION_COMP_DOWNGRADE_ID: _COMMON_UPDATE_SOLUTION_COMP_RES,
   USER_COMP_BASEIMAGE_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   USER_COMP_ADDON_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   USER_COMP_HSP_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   USER_COMP_SOLUTION_DOWNGRADE_ID: _COMMON_UPDATE_SOLUTION_COMP_RES,
   VIB_DOWNGRADE_ID: "Add vendor Addon and Components that include matching " \
      "or higher version VIBs to the image.",

   ADDON_REMOVAL_ID: "Add the vendor Addon to the image to retain it.",
   COMPONENT_REMOVAL_ID: "Add the Components to the image to retain them. " \
      "If this is seen while switching from using Baselines to using Images, "\
      "please refer to KB 90188.",
   HSP_REMOVAL_ID: "Enable the corresponding Hardware Support Manager to " \
                   "retain the package.",
   SOLUTIONCOMPONENT_REMOVAL_ID: "Enable corresponding functionalities of " \
      "the Solution to add the Components back to the image.",
   VIB_REMOVAL_ID: "To prevent them from being removed, include appropriate " \
      "components that are equivalent to these VIBs. If this is seen while " \
      "switching from using Baselines to using Images, please refer to " \
      "KB 90188.",

   UNSUPPORTED_VENDOR_MODEL_ID % 'Vendor': "Replace vendor Addon or " \
      "Components that are released for a different hardware vendor.",
   UNSUPPORTED_VENDOR_MODEL_ID % 'Model': "Replace vendor Addon or " \
      "Components that are released for a different hardware model.",
   NO_NIC_DRIVER_ID: "Make sure the image contains the appropriate driver " \
      "for the management NIC.",
   NO_BOOTDEVICE_DRIVER_ID: "Make sure the image contains the appropriate " \
      "driver for the boot device.",
   INSUFFICIENT_BOOTBANK_SPACE_ID: "Remove unnecessary Components from " \
      "the image.",
   INSUFFICIENT_LOCKER_SPACE_ID: "Cleanup the locker partition on the host, " \
      "and retry check compliance.",
   DISK_TOO_SMALL_ID: "",
   HOST_ACCEPTANCE_ID: "Make sure the host is configured at an appropriate " \
      "acceptance level for the image, and retry check compliance.",
   VSAN_V1_DISK_ID: "Make sure all vSAN V1 disks have been upgraded to latest" \
      " disk format version.",
   UNSUPPORTED_TPM_ID: "Make sure the host is upgraded to TPM 2.0.",
   MISSING_GUNZIP_CHECKSUM_VIB_ID: "Please remove these VIBs and" \
      " check with your vendor for a replacement of these VIBs",
   INSUFFICIENT_ENTROPY_CACHE: "Refer to KB 89854 for steps on how to refill " \
      "the cache.",
   INCOMPAT_SPHERELET_VER_ID: "Please upgrade your WCP cluster to install a " \
      "compatible Spherelet version, or remove Spherelet if the host is not " \
      "in a WCP cluster.",
   UNSUPPORTED_SHA1_CERT: "Support for certificates with weak signature " \
      "algorithm SHA-1 has been removed in ESXi 8.0. To proceed with upgrade," \
      " replace it with a SHA-2 signature based certificate.",
   COMPONENTS_REMOVED: "Apply equal or higher versions of the Component(s) " \
      "on the host, and retry depot extraction.",
   ORPHAN_VIB: "Apply Component(s) that contain these VIB(s) for them to be " \
      "included in the extracted depot.",
   SEEDING_PENDING_REBOOT_ID: "Reboot the host to use the latest software " \
      "change.",
   UNSUPPORTED_STATELESS_HOST_ID: "Retry getting software info and depot " \
      "extraction on a stateful host.",
   VFAT_CORRUPTION: "Please refer to KB 91136 and run dosfsck on " \
                    "all bootbank partitions.",

   # Live Patch specific.
   QP_TPM_UNSUPPORTED_ID: "Disable the 'Enforce Live Patch' remediation " \
      "option and retry the operation without using Live Patch.",
   QP_DPU_UNSUPPORTED_ID: "Disable the 'Enforce Live Patch' remediation " \
      "option and retry the operation without using Live Patch.",
   QP_LIVE_COMP_UNSUPPORTED_ID: "Disable the 'Enforce Live Patch' " \
      "remediation option and retry the operation without using Live Patch. " \
      "Alternatively, modify the image and remove the drifts in these " \
      "component(s), then retry the operation.",
   QP_LIVE_ORPHAN_VIB_UNSUPPORTED_ID: "Disable the 'Enforce Live Patch' " \
      "remediation option and retry the operation.",
   QP_LIVE_SOLUTION_UNSUPPORTED_ID: "Disable the 'Enforce Live Patch' " \
      "remediation option and retry the operation without using Live Patch. " \
      "Alternatively, disable the remediation option, modify the image to " \
      "remediate these solution(s) first, then re-enable the option and " \
      "retry Live Patch.",
   QP_SCANSCRIPT_FAILURE_ID: "Contact VMware support for assistance, or " \
      "disable the 'Enforce Live Patch' remediation option and retry the " \
      "operation without using Live Patch.",
   }

# Map from precheck result name to notification ID.
PRECHECK_NOTIFICATION_ID = {
   "MEMORY_SIZE": INSUFFICIENT_MEMORY_ID,
   "CPU_SUPPORT": UNSUPPORTED_CPU_ID,
   "CPU_CORES": NOT_ENOUGH_CPU_CORE_ID,
   "BIOS_FIRMWARE_TYPE": UNSUPPORTED_BIOS_FIRMWARE_ID,
   "UNSUPPORTED_DEVICES": UNSUPPORTED_PCI_DEVICE,
   "VALIDATE_HOST_HW": UNSUPPORTED_VENDOR_MODEL_ID,
   "NATIVE_BOOT_NIC": NO_NIC_DRIVER_ID,
   "NATIVE_BOOTBANK": NO_BOOTDEVICE_DRIVER_ID,
   "IMAGEPROFILE_SIZE": INSUFFICIENT_BOOTBANK_SPACE_ID,
   "LOCKER_SPACE_AVAIL": INSUFFICIENT_LOCKER_SPACE_ID,
   "BOOT_DISK_SIZE": DISK_TOO_SMALL_ID,
   "HOST_ACCEPTANCE": HOST_ACCEPTANCE_ID,
   "VSAN_V1_DISK": VSAN_V1_DISK_ID,
   "TPM_VERSION": UNSUPPORTED_TPM_ID,
   "MISSING_GUNZIP_CHECKSUM_VIBS": MISSING_GUNZIP_CHECKSUM_VIB_ID,
   "SPHERELET_VERSION": INCOMPAT_SPHERELET_VER_ID,
   "SHA1_CERT": UNSUPPORTED_SHA1_CERT,
   "ENTROPY_CACHE": INSUFFICIENT_ENTROPY_CACHE,
   "VFAT_CORRUPTION": VFAT_CORRUPTION,
   }

# Compliance status to compliance details ID mapping.
HOST_COMPLIANCE_DETAILS_ID = {
   UNAVAILABLE: HOST_COMPLIANCE_UNKNOWN_ID,
   INCOMPATIBLE: HOST_COMPLIANCE_INCOMPATIBLE_ID,
   NON_COMPLIANT: HOST_COMPLIANCE_NONCOMPLIANT_ID,
   COMPLIANT: HOST_COMPLIANCE_COMPLIANT_ID,
}

DPU_COMPLIANCE_DETAILS_ID = {
   UNAVAILABLE: DPU_COMPLIANCE_UNKNOWN_ID,
   INCOMPATIBLE: DPU_COMPLIANCE_INCOMPATIBLE_ID,
   NON_COMPLIANT: DPU_COMPLIANCE_NONCOMPLIANT_ID,
   COMPLIANT: DPU_COMPLIANCE_COMPLIANT_ID,
}

# Compliance status to compliance details notification type mapping.
COMPLIANCE_TO_NOTIFICATION_TYPE = {
   UNAVAILABLE: ERROR,
   INCOMPATIBLE: ERROR,
   NON_COMPLIANT: WARNING,
   COMPLIANT: INFO,
}

# Image entities.
BASE_IMAGE = "base_image"
ADDON = "addon"
COMPONENT = "component"
SOLUTION = "solution"
BASE_IMAGE_COMP = BASE_IMAGE + '_' + COMPONENT
ADDON_COMP = ADDON + '_' + COMPONENT
HARDWARE_SUPPORT_COMP = HARDWARE_SUPPORT + '_' + COMPONENT
SOLUTION_COMPONENT = SOLUTION + '_' + COMPONENT
OLD_ADDON_COMP = "old_addon_component"

# Convert component sources to image entities.
SOURCE_TO_ENTITY = {
   SOURCE_BASEIMAGE: BASE_IMAGE,
   SOURCE_ADDON: ADDON,
   SOURCE_HSP: HARDWARE_SUPPORT,
   SOURCE_SOLUTION: SOLUTION,
   SOURCE_USER: COMPONENT,
   }

# Map from downgrade entity to notification ID.
DOWNGRADE_NOTIFICATION_ID = {
   BASE_IMAGE: BASEIMAGE_DOWNGRADE_ID,
   ADDON: ADDON_DOWNGRADE_ID,
   COMPONENT: COMPONENT_DOWNGRADE_ID,
   HARDWARE_SUPPORT: HSP_DOWNGRADE_ID,
   HARDWARE_SUPPORT_COMP: HSP_COMP_DOWNGRADE_ID,
   }

# Base Image components that can be removed by the user.
BASE_IMAGE_REMOVABLE_COMPS = ('VMware-Host-Client', 'VMware-VM-Tools')

# Component downgrade notification IDs indexed by from and to component source.
COMP_DOWNGRADE_NOTIFICATION_ID = {
   BASE_IMAGE: {
      BASE_IMAGE: BASEIMAGE_COMP_DOWNGRADE_ID,
      # BaseImage is always present so unless a component is removed from
      # BaseImage at one point, there should not be BaseImage to Addon/HSP
      # downgrades.
   },
   ADDON: {
      BASE_IMAGE: ADDON_COMP_BASEIMAGE_DOWNGRADE_ID,
      ADDON: ADDON_COMP_DOWNGRADE_ID,
      # When Addon is being removed, HSP might provide the same component.
      HARDWARE_SUPPORT: ADDON_COMP_HSP_DOWNGRADE_ID,
   },
   HARDWARE_SUPPORT: {
      BASE_IMAGE: HSP_COMP_BASEIMAGE_DOWNGRADE_ID,
      ADDON: HSP_COMP_ADDON_DOWNGRADE_ID,
      HARDWARE_SUPPORT: HSP_COMP_DOWNGRADE_ID,
   },
   SOLUTION: {
      # Solution components should not be found in another release unit.
      SOLUTION: SOLUTION_COMP_DOWNGRADE_ID,
   },
   COMPONENT: {
      # User component to release unit downgrades are reported by release
      # units.
      BASE_IMAGE: USER_COMP_BASEIMAGE_DOWNGRADE_ID,
      ADDON: USER_COMP_ADDON_DOWNGRADE_ID,
      HARDWARE_SUPPORT: USER_COMP_HSP_DOWNGRADE_ID,
      SOLUTION: USER_COMP_SOLUTION_DOWNGRADE_ID,
   }
}

# Image profile scan spec.
BASEIMAGE_COMPS_KEY = BASE_IMAGE_COMP + 's'
ADDON_COMPS_KEY = ADDON_COMP + 's'
# USER_COMP_KEY references all the components (Added, Upgraded or Downgraded) by
# the user
USER_COMPS_KEY = "user_components"

ORPHAN_VIBS = 'orphan_vibs'

# Below are for image customization
# USER_OVERRIDDEN_COMPS_KEY references only the newly Added, Upgraded or
# Downgraded components by the user
USER_OVERRIDDEN_COMPS_KEY = "user_overriden_components"
DOWNGRADED_BI_COMP_KEY = "downgraded_bi_components"
REMOVED_BI_COMP_KEY = "removed_bi_components"
DOWNGRADED_ADDON_COMP_KEY = "downgraded_add_on_components"
REMOVED_ADDON_COMP_KEY = "removed_add_on_components"
# Keys for user removed components.
USER_REMOVED_BI_COMP_KEY = "user_removed_bi_components"
USER_REMOVED_ADDON_COMP_KEY = "user_removed_addon_components"

IMAGE_CUSTOMIZATION_ACTION_REMOVED = "REMOVED"
IMAGE_CUSTOMIZATION_ACTION_RETAINED = "RETAINED"
_IMAGE_CUSTOMIZATION_PREFIX = \
   "com.vmware.vcIntegrity.lifecycle.softwarespec.ImageCustomization."
IMAGE_CUSTOMIZATION_RM_COMP_ID = _IMAGE_CUSTOMIZATION_PREFIX + \
                                 "ComponentRemoved"
# On UI, the removed version of the component will be striked out, and the
# localized version of "REMOVED" will be displayed as the version.
IMAGE_CUSTOMIZATION_RM_COMP_MSG = "REMOVED"
IMAGE_CUSTOMIZATION_RT_COMP_ID = _IMAGE_CUSTOMIZATION_PREFIX + \
                                 "ComponentRetained"
IMAGE_CUSTOMIZATION_RT_COMP_MSG = "RETAINED"

# Validate software spec entities.
VALIDATE_PREFIX = 'com.vmware.vcIntegrity.lifecycle.Validate.'
VALIDATE_SUCCESS_ID = VALIDATE_PREFIX + 'Success'
VALIDATE_SUCCESS_MSG = 'The image is valid.'

COMP_RETENTION_WARNING_ID = VALIDATE_PREFIX + 'Retention'
COMP_RETENTION_WARNING_MSG = "Component {1} will be retained at an older " \
   "version {2} instead of {3} from {4}. This could be lower than the " \
   "supported version."
COMP_RETENTION_RES_ID = COMP_RETENTION_WARNING_ID + '.Resolution'
COMP_RETENTION_RES_MSG = "Test the desired state before applying it to large " \
   "number of servers. Refer to vLCM documentation for more details."

COMP_REMOVAL_WARNING_ID = VALIDATE_PREFIX + 'Removal'
COMP_REMOVAL_WARNING_MSG = "Component {1} version {2} from {3} will be removed."
COMP_REMOVAL_RES_ID = COMP_REMOVAL_WARNING_ID + '.Resolution'
COMP_REMOVAL_RES_MSG = "Test the desired state before applying it to large " \
   "number of servers. Refer to vLCM documentation for more details."

# Depot metadata schema validation entities
DEPOT_METADATA_SCHEMA_VALIDATE_SUCCESS_MSG = \
           'Depot validation succeeded.'

# Live Patch no pending actions.
QUICK_PATCH_NO_ACTIONS = ('No remediation action is required during the '
                          'transaction.')
QUICK_PATCH_REMEDIATION_DEFAULT_MSG = ('All remediation actions have been '
                                       'completed.')
