# Copyright 2019-2021,2023 VMware, Inc.
# All rights reserved. -- VMware Confidential

"""Constants
Constants that are used in the ImageManager.
"""

# Scanner Constants
# Status
COMPLIANT = "COMPLIANT"
NON_COMPLIANT = "NON_COMPLIANT"
INCOMPATIBLE = "INCOMPATIBLE"
UNAVAILABLE = "UNAVAILABLE"

# Impact types and messages.
IMPACT_NONE = "NO_IMPACT"
IMPACT_MMODE = "MAINTENANCE_MODE_REQUIRED"
IMPACT_REBOOT = "REBOOT_REQUIRED"

# Impact from WCP/NSX components needs to be re-evaluated
NSX_COMPONENT = "nsx-lcp-bundle"
WCP_COMPONENT_PREFIX = "VMware-Spherelet-"

# Partial Maintenance Mode types
PARTIAL_MAINTENANCE_MODE_WCP = "sphereletPartialMM"

# Component source, see SoftwareInfo.vmodl.
SOURCE_BASEIMAGE = "BASE_IMAGE"
SOURCE_ADDON = "ADD_ON"
SOURCE_HSP = "HARDWARE_SUPPORT_PACKAGE"
SOURCE_SOLUTION = "SOLUTION"
SOURCE_USER = "USER"

# Spec constants
DISP_VERSION = "display_version"
DISP_NAME = "display_name"
VERSION = "version"
VENDOR = "vendor"
BASE_IMG = "base_image"
BASEIMAGE_UI_NAME = "ESXi"
ADD_ON = "add_on"
COMPONENTS = "components"
HARDWARE_SUPPORT = "hardware_support"
SOLUTIONS = "solutions"
CONFIG_SCHEMA = 'config_schema'

# Notification ID and messages.
HOSTSCAN_PREFIX = "com.vmware.vcIntegrity.lifecycle.HostScan."
HOSTSEEDING_PREFIX = "com.vmware.vcIntegrity.lifecycle.HostSeeding."
HOST_PREFIX = "com.vmware.vcIntegrity.lifecycle.Host."
RESOLUTION_SUFFIX = ".Resolution"

UNAVAILABLE_ID = HOSTSCAN_PREFIX + "Unavailable"

#  Component downgrades.
BASEIMAGE_DOWNGRADE_ID = HOSTSCAN_PREFIX + "BaseImageDowngrade"
BASEIMAGE_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "BaseImageComponentDowngrade"
ADDON_DOWNGRADE_ID = HOSTSCAN_PREFIX + "AddonDowngrade"
ADDON_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "AddonComponentDowngrade"
ADDON_COMP_BASEIMAGE_DOWNGRADE_ID = \
   HOSTSCAN_PREFIX + "AddonCompBaseImageDowngrade"
ADDON_COMP_HSP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "AddonCompHSPDowngrade"
COMPONENT_DOWNGRADE_ID = HOSTSCAN_PREFIX + "ComponentDowngrade"
HSP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPDowngrade"
HSP_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPComponentDowngrade"
HSP_COMP_BASEIMAGE_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPCompBaseImageDowngrade"
HSP_COMP_ADDON_DOWNGRADE_ID = HOSTSCAN_PREFIX + "HSPCompAddonDowngrade"
SOLUTION_COMP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "SolutionComponentDowngrade"
USER_COMP_BASEIMAGE_DOWNGRADE_ID = \
   HOSTSCAN_PREFIX + "UserCompBaseImageDowngrade"
USER_COMP_ADDON_DOWNGRADE_ID = HOSTSCAN_PREFIX + "UserCompAddonDowngrade"
USER_COMP_HSP_DOWNGRADE_ID = HOSTSCAN_PREFIX + "UserCompHSPDowngrade"
USER_COMP_SOLUTION_DOWNGRADE_ID = HOSTSCAN_PREFIX + "UserCompSolutionDowngrade"
VIB_DOWNGRADE_ID = HOSTSCAN_PREFIX + "VibDowngrade"

ADDON_REMOVAL_ID = HOSTSCAN_PREFIX + "AddonRemoval"
COMPONENT_REMOVAL_ID = HOSTSCAN_PREFIX + "ComponentRemoval"
HSP_REMOVAL_ID = HOSTSCAN_PREFIX + "HSPRemoval"
SOLUTIONCOMPONENT_REMOVAL_ID = HOSTSCAN_PREFIX + "SolutionComponentRemoval"
VIB_REMOVAL_ID = HOSTSCAN_PREFIX + "VibRemoval"

SOLUTION_DISABLE_ID = HOSTSCAN_PREFIX + "SolutionDisable"
SOLUTION_ENABLE_ID = HOSTSCAN_PREFIX + "SolutionEnable"

BASEIMAGE_COMPONENT_REMOVED_ID = HOSTSCAN_PREFIX + "BaseImageComponentRemoved"
ADDON_COMPONENT_REMOVED_ID = HOSTSCAN_PREFIX + "AddonComponentRemoved"
HSP_COMPONENT_REMOVED_ID = HOSTSCAN_PREFIX + "HSPComponentRemoved"

MAINTMODE_IMPACT_ID = HOSTSCAN_PREFIX + "MaintenanceModeImpact"
REBOOT_IMPACT_ID = HOSTSCAN_PREFIX + "RebootImpact"
PENDING_REBOOT_ID = HOSTSCAN_PREFIX + "PendingReboot"

INSUFFICIENT_MEMORY_ID = HOSTSCAN_PREFIX + "InsufficientMemory"

UNSUPPORTED_CPU_ID = HOSTSCAN_PREFIX + "UnsupportedCpu"
UNSUPPORTED_CPU_WARN_ID = HOSTSCAN_PREFIX + "UnsupportedCpuWarn"
UNSUPPORTED_CPU_OVERRIDE_ID = HOSTSCAN_PREFIX + "UnsupportedCpuOverride"

NOT_ENOUGH_CPU_CORE_ID = HOSTSCAN_PREFIX + "NotEnoughCpuCore"
UNSUPPORTED_PCI_DEVICE = HOSTSCAN_PREFIX + "UnsupportedPciDevice"
# Unsupported Model or Vendor
UNSUPPORTED_VENDOR_MODEL_ID = HOSTSCAN_PREFIX + "Unsupported%s"
NO_NIC_DRIVER_ID = HOSTSCAN_PREFIX + "NoNicDriver"
NO_BOOTDEVICE_DRIVER_ID = HOSTSCAN_PREFIX + "NoBootDeviceDriver"
INSUFFICIENT_BOOTBANK_SPACE_ID = HOSTSCAN_PREFIX + "InsufficientBootBankSpace"
INSUFFICIENT_LOCKER_SPACE_ID = HOSTSCAN_PREFIX + "InsufficientLockerSpace"
DISK_TOO_SMALL_ID = HOSTSCAN_PREFIX + "DiskTooSmall"
HOST_ACCEPTANCE_ID = HOSTSCAN_PREFIX + "HostAcceptance"
VSAN_V1_DISK_ID = HOSTSCAN_PREFIX + "VsanV1Disk"

#Host seeding related
COMPONENTS_REMOVED_DG = HOSTSEEDING_PREFIX + "ComponentsRemovedDowngraded"
ORPHAN_VIB = HOSTSEEDING_PREFIX + "OrphanVib"
SEEDING_PENDING_REBOOT_ID = HOSTSEEDING_PREFIX + "PendingReboot"
UNSUPPORTED_STATELESS_HOST_ID = HOSTSEEDING_PREFIX + "UnsupportedStatelessHost"

# QuickBoot compatibility check
DRIVER_MAP_DIRS = set([
   "etc/vmware/default.map.d",
   "etc/vmware/fallback.map.d",
   "etc/vmware/ihv.map.d",
   "etc/vmware/oem.map.d",
])

_QUICKBOOT_PREFIX = HOSTSCAN_PREFIX + "QuickBoot."
QUICKBOOT_UNSUPPORTED_ID = _QUICKBOOT_PREFIX + "Unsupported"
QUICKBOOT_UNAVAILABLE_DRIVER_DG_ID = \
   _QUICKBOOT_PREFIX + "Unavailable.DriverDowngrade"
QUICKBOOT_SUPPORTED_ID = _QUICKBOOT_PREFIX + "Supported"

# QuickBoot and SuspendToMemory compatibility check
_QUICKBOOT_AND_SUSPEND_TO_MEMORY_PREFIX = \
   HOSTSCAN_PREFIX + "QuickBootAndSuspendToMemory."
QUICKBOOT_AND_SUSPEND_TO_MEMORY_SUPPORTED_ID = \
   _QUICKBOOT_AND_SUSPEND_TO_MEMORY_PREFIX + "Supported"

# Error message IDs, default messages and arguments.
ESXIMAGE_PREFIX = "com.vmware.vcIntegrity.lifecycle.EsxImage."

#  Usually the name of the exeption will be the key in APPLY_EXCEPTION_MSG_ARG,
#  here aliases are included to convert an exception.
ESXIMAGE_ERROR_ALIAS = {
   "ComponentConfigDowngrade": "ComponentDowngradeError",
   # Profile validation contains a few other checks, but generally it is
   # correct to say the components have validation issues.
   "ProfileValidationError": "ComponentValidationError",
   "DependencyError": "ComponentValidationError",
}

NOTIFICATION_MSG = {
   UNAVAILABLE_ID: "Host compliance check failed with an unknown error.",
   BASEIMAGE_DOWNGRADE_ID: "ESXi version {1} in the image is a downgrade of " \
      "the installed version {2}.",
   BASEIMAGE_COMP_DOWNGRADE_ID: "Downgrades of ESXi Components {1} are not " \
      "supported.",
   ADDON_DOWNGRADE_ID: "Version {1} of vendor Addon {2} in the image is a " \
      "downgrade of the installed version {3}.",
   ADDON_COMP_DOWNGRADE_ID: "Downgrades of Addon Components {1} in the " \
      "desired Addon {2} are not supported.",
   ADDON_COMP_BASEIMAGE_DOWNGRADE_ID: "Downgrades of Addon Components {1} in " \
      "the desired ESXi version are not supported.",
   ADDON_COMP_HSP_DOWNGRADE_ID: "Downgrades of Addon Components {1} in the " \
      "desired Hardware Support Package {2} are not supported.",
   HSP_DOWNGRADE_ID: "Version {1} of Hardware Support Package {2} in the " \
      "image is a downgrade of the installed version {3}.",
   HSP_COMP_DOWNGRADE_ID: "Downgrades of Hardware Support Package Components " \
      "{1} in the desired Hardware Support Package {2} are not supported.",
   HSP_COMP_BASEIMAGE_DOWNGRADE_ID: "Downgrades of Hardware Support Package " \
      "Components {1} in the desired ESXi version are not supported.",
   HSP_COMP_ADDON_DOWNGRADE_ID: "Downgrades of Hardware Support Package " \
      "Components {1} in the desired Addon {2} are not supported.",
   COMPONENT_DOWNGRADE_ID: "Version {1} of the manually added Component {2} " \
      "is an unsupported downgrade from version {3} on the host.",
   SOLUTION_COMP_DOWNGRADE_ID: "Downgrades of Components {1} in Solution {2} " \
      "are not supported.",
   USER_COMP_BASEIMAGE_DOWNGRADE_ID: "Downgrades of manually added " \
      "Components {1} in the desired ESXi version are not supported.",
   USER_COMP_ADDON_DOWNGRADE_ID: "Downgrades of manually added Components " \
      "{1} in the desired Addon {2} are not supported.",
   USER_COMP_HSP_DOWNGRADE_ID: "Downgrades of manually added Components {1} " \
      "in the desired Hardware Support Package {2} are not supported.",
   USER_COMP_SOLUTION_DOWNGRADE_ID: "Downgrades of manually added Components " \
      "{1} in the desired Solution {2} are not supported.",
   VIB_DOWNGRADE_ID: "Downgrades of host VIBs {1} in the image are not " \
      "supported.",
   ADDON_REMOVAL_ID: "Vendor Addon {1} with version {2} is removed in the " \
      "image. It will be removed from the host during remediation.",
   HSP_REMOVAL_ID: "Hardware Support Package {1} with version {2} is " \
      "removed in the image. It will be removed from the host during " \
      "remediation.",
   COMPONENT_REMOVAL_ID: "Components {1} are removed in the image. They will " \
      "be removed from the host during remediation.",
   SOLUTIONCOMPONENT_REMOVAL_ID: "Components {1} in Solution {2} are removed " \
      "in the image. They will be removed from the host during remediation.",
   VIB_REMOVAL_ID: "The following VIBs on the host are missing from the " \
      "image and will be removed from the host during remediation: {1}.",

   SOLUTION_DISABLE_ID: "Solution Components of disabled Solutions {1} will " \
                        "be removed from this host during remediation.",
   SOLUTION_ENABLE_ID: "Remediating this host will enable these Solutions: " \
                       "{1}.",

   BASEIMAGE_COMPONENT_REMOVED_ID: "ESXi Components {1} are either missing " \
      "or downgraded on the host. Appropriate action will be taken during " \
      "remediation.",
   ADDON_COMPONENT_REMOVED_ID: "Vendor Addon Components {1} are either " \
      "missing or downgraded on the host. Appropriate action will be taken " \
      "during remediation.",
   HSP_COMPONENT_REMOVED_ID: "Hardware Support Package Components {1} are " \
      "either missing or downgraded on the host. The correct versions of " \
      "the Components will be installed during remediation.",

   MAINTMODE_IMPACT_ID: "The host will be placed in maintenance mode before " \
      "remediation.",
   REBOOT_IMPACT_ID: "The host will be rebooted during remediation.",
   PENDING_REBOOT_ID: "The host is pending a reboot since a previous " \
                      "software change. The pending change is not reflected " \
                      "in host's current state in the compliance results " \
                      "below. Reboot the host to see accurate compliance " \
                      "results.",

   INSUFFICIENT_MEMORY_ID: "The host has {1} MB of memory, a minimum of {2} " \
      "MB is required by the image.",
   UNSUPPORTED_CPU_ID: "The CPU on the host is not supported by the image.",
   UNSUPPORTED_CPU_WARN_ID: "The CPU in this host may not be supported in " \
      "future ESXi releases. Please plan accordingly.",
   UNSUPPORTED_CPU_OVERRIDE_ID: "The CPU in this host is not supported by ESXi." \
      " You can override and force remediate, but it is neither officially supported nor recommended.",
   NOT_ENOUGH_CPU_CORE_ID: "The CPU on the host has {1} cores, a minimum of " \
      "{2} cores are required by the image.",
   UNSUPPORTED_PCI_DEVICE: "Unsupported PCI devices {1} found on the host.",
   UNSUPPORTED_VENDOR_MODEL_ID % 'Vendor': "Host hardware vendor {1} is not " \
      "supported by the image.",
   UNSUPPORTED_VENDOR_MODEL_ID % 'Model': "Host hardware model {1} is not " \
      "supported by the image.",
   NO_NIC_DRIVER_ID: "The management NIC of the host lacks a driver in the " \
      "image.",
   NO_BOOTDEVICE_DRIVER_ID: "The boot device of the host lacks a driver in " \
      "the image.",
   INSUFFICIENT_BOOTBANK_SPACE_ID: "The bootbank partition on the host has a " \
      "capacity of {1} MB, the image requires {2} MB.",
   INSUFFICIENT_LOCKER_SPACE_ID: "The locker partition on the host has {1} " \
      "MB free space available, the image requires {2} MB.",
   DISK_TOO_SMALL_ID: "The boot device of the host has a capacity of {1} MB, " \
      "a minimum of {2} MB is required by the image.",
   HOST_ACCEPTANCE_ID: "Acceptance level of the image is not " \
      "compatible with the host's configured setting.",

   QUICKBOOT_UNSUPPORTED_ID: "Quick Boot is not supported on the host.",
   QUICKBOOT_UNAVAILABLE_DRIVER_DG_ID: "Quick Boot is not available due to " \
      "downgrade of driver(s): {1}.",
   QUICKBOOT_SUPPORTED_ID: "Quick Boot is supported on the host.",
   QUICKBOOT_AND_SUSPEND_TO_MEMORY_SUPPORTED_ID: "Quick Boot and "
      "Suspend To Memory are supported on the host.",
   VSAN_V1_DISK_ID: "vSAN v1 disks are not supported for upgrade.",

   COMPONENTS_REMOVED_DG: "The following Components have been " \
      "removed or downgraded on the host: {1}",
   ORPHAN_VIB: "The following standalone VIB(s) are present on the host " \
      "and will not be extracted: {1}",
   SEEDING_PENDING_REBOOT_ID: "The host is pending a reboot since a " \
      "previous software change. The pending change is not reflected " \
      "in the software info and will not be extracted into the depot.",
   UNSUPPORTED_STATELESS_HOST_ID: "Getting software info and depot " \
      "extraction are not supported on a stateless host.",
   }

#  Map from name of exception to default message and arguments to extract from
#  the exception.
ESXIMAGE_ERROR_MSG_ARG = {
   "AcceptanceConfigError": ("A VIB package cannot be installed due to the "
                             "current host acceptance level configuration.",
                             []),
   "AcceptanceGetError": ("Failed to get the current host acceptance level "
                          "configuration.",
                          []),
   "AddonBaseImageMismatchError": ("Addon {1} is incompatible with base "
                                   "image version {2}.",
                                   ["addonId", "baseImageVersion"]),
   "AddonNotFound": ("Addon {1} with version {2} cannot be found in depot.",
                     ["name", "version"]),
   "AddonValidationError": ("Failed to validate metadata or components of " \
                            "addon {1}.",
                            ['addonID']),
   "BaseImageNotFound": ("The Base Image with version {1} cannot be found in "
                         "depot.",
                         ["version"]),
   "BaseimageValidationError": ("Failed to validate metadata or components " \
                                "of the base image.",
                                []),
   "ComponentDowngradeError": ("The operation cannot continue due to downgrade "
                               "of the following Components: {1}.",
                               ["components"]),
   "ComponentFormatError": ("An error occurred while parsing or merging "
                            "components.",
                            []),
   "ComponentNotFoundError": ("Component {1} cannot be found in depot.",
                              ["name"]),
   "ComponentUnmountError": ("Failed to remove Component {1}, files may "
      "still be in use.",
      ["component"]),
   "ComponentValidationError": ("There are Compoment relation and content "
                                "problems with the desired state.",
                                []),
   "DatabaseIOError": ("Cannot read or write the image database on the host.",
                       []),
   "DatabaseFormatError": ("The image database on the host is invalid or "
                           "corrupted.",
                           []),
   "DepotConnectError": ("An error occurred while connecting to depot.",
                         []),
   "HardwareSupportPackageNotFound": (
      "Manifest with Hardware Support Manager {1} and Hardware Support "
      "Package {2} version {3} cannot be found in depot.",
      ["manager", "package", "version"]),
   "IncompatibleSolutionCompsError": ("No compatible combination of Components "
      "{1} is found in Solution {2}.",
      ["compNames", "solutionName"]),
   "IncompatibleSolutionComponentError": ("No compatible Component {1} is "
      "found in Solution {2}, candidate versions are: {3}.",
      ["name", "solutionName", "compVersions"]),
   "InstallationError": ("An internal error occurred "
                         "while remediating the host.",
                         []),
   "LiveInstallationError": ("An error occurred while performing live system "
                             "changes during the remediation.",
                             []),
   "DataStorageNotFound": ("No OSData storage partition is available to "
      "extract depot. Configure persistent storage for the host and retry.",
      []),
   "LockingError": ("Remediation cannot begin because another process is "
                    "updating the host.",
                    []),
   "MaintenanceModeError": ("Maintenance mode is either not enabled or cannot "
                            "be determined on the host.",
                            []),
   "MemoryReserveError": (
      "The host is out of resource: failed to reserve {1} MB of memory during "
      "remediation.",
      ["reserveSize"]),
   "ManifestBaseImageMismatchError": (
      "Manifest with Hardware Support Manager {1} and Hardware Support "
      "Package {2} version {3} does not support Base Image version {4}.",
      ["manager", "package", "version", "baseImageVersion"]),
   "ManifestConflictError":(
      "Hardware Support Packages with ID {1} and {2} have overlapping "
      "Base Image support. Use a different package that does not overlap "
      "with package {1}.",
      ["manifestID", "newManifestID"]),
   "MetadataDownloadError": ("An error occurred while downloading depot "
                             "metadata from {1}.",
                             ['url']),
   "MetadataFormatError": ("The format of metadata in {1} is invalid.",
                           ['filename']),
   "MetadataIOError": ("An error occured while reading metadata.zip.",
                       []),
   "MissingComponentError": ("Cannot find metadata for a Component.",
                             []),
   "MissingVibError": ("Cannot find metadata for VIB(s) {1} in database or "
                       "depot.",
                       ["vibIDs"]),
   "MultipleManifestError": (
      "The current version of ESXi does not support multiple Hardware Support "
      "Packages.",
      []),
   "ReleaseUnitConflictError": ("There are unequal Base Images, Addons or "
                                "Solutions metadata that share the "
                                "same ID {1}.",
                                ['releaseID']),
   "ReservedVibExtractError": ("Failed to extract depot from the host: no "
                               "stored copy available for inactive VIB {1}.",
                               ['vib']),

   "ServiceDisableError": ("Failed to remove Component {1}: an error "
      "occurred while disabling service {2}.",
      ["component", "service"]),
   "ServiceEnableError": ("Failed to apply Component {1}: an error "
      "occurred while enabling service {2}.",
      ["component", "service"]),
   "SoftwareInfoExtractError": (
      "An error occurred while extracting image info on the host.",
      ['msg']),
   "SoftwareSpecFormatError": ("Corrputed or invalid software specification "
                               "format. Missing attribute: {1}.",
                               ['field']),
   "SolutionComponentNotFound": ("Component(s) {1} within software Solution {2}"
                                 " cannot be found in depot.",
                                 ["compNames", "solutionName"]),
   "SolutionNotFound": ("Software Solution {1} with version {2} cannot be "
                        "found in depot.",
                        ["name", "version"]),
   # vLCM supports only stateful hosts, StatelessError means
   # there is an issue with /bootbank or /altbootbank.
   "StatelessError": ("The bootbank partition is not accessible " \
                      "or contains an invalid image database. " \
                      "Check /bootbank, /altbootbank paths "\
                      "are valid and accessible.",
                      []),
   "VibCertStoreError": ("Failed to import certificates or revocation lists "
                         "while verifying VIB signature.",
                         []),
   "VibChecksumError": ("Recreation of the VIB {1} failed due to checksum "
                        "mismatch. VIB may have been created using older "
                        "tools. Try a different host or update the component "
                        "that contains this VIB.",
                        ['vib']),
   "VibDownloadError": ("An error occurred while downloading VIB from URL {1}.",
                        ['url']),
   "VibFormatError": ("The format of VIB {1} is invalid.",
                      ['filename']),
   "VibHwPlatformError": ("Hardware vendor and model requirements of "
                          "VIBs cannot be satisfied by this host. Use VIBs "
                          "that match hardware vendor and model of host.",
                          []),
   "VibRecreateError": ("Failed to extract depot from the host: an error "
                        "occurred while re-creating VIB {1}.",
                        ['vib']),
   "VibSigDigestError": ("Signature of VIB {1} cannot be verified against its "
                         "contents.",
                         ['vib']),
   "VibSigFormatError": ("VIB {1} contains an invalid or corrupted signature.",
                         ['vib']),
   "VibSigInvalidError": ("VIB {1} contains an invalid or corrupted signature.",
                          ['vib']),
   "VibSigMissingError": ("VIB {1} does not have a signature and thus cannot "
                          "be verified.",
                          ['vib']),
   "VibXmlSchemaLoadError": ("Failed to load XML schema for VIB validation. "
                             "Check if XML schema file {1} is present on the host.",
                             ['filename']),
   # Catch-all unknown error message that need to be used when the type is not
   # listed above.
   "UnknownError": ("An unknown error occurred while performing the operation.",
                    []),
   }

# Resolution messages, indexed by notification/message IDs.
#   When an ID does not have a message here, the resolution structure of the
#   notification will automatically have empty ID and message.
_COMMON_ADD_USERCOMP_RES = "Add the Components of matching or " \
   "higher versions to the image."
_COMMON_UPDATE_SOLUTION_COMP_RES = "Check the Solution's compatibility " \
   "matrix to enable higher version Component(s), or disable the Solution."
RESOLUTION_MSG = {
   UNAVAILABLE_ID: "Retry check compliance at a later time.",
   BASEIMAGE_DOWNGRADE_ID: "Replace the ESXi version in the image with one " \
      "of the same or higher version.",
   BASEIMAGE_COMP_DOWNGRADE_ID: "Select a Base Image that does not downgrade " \
      "an installed Base Image Component.",
   ADDON_DOWNGRADE_ID: "Replace the vendor Addon in the image with one of " \
      "the same or higher version.",
   ADDON_COMP_DOWNGRADE_ID: "Select an vendor Addon that does not downgrade " \
      "an installed Addon Component.",
   ADDON_COMP_BASEIMAGE_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   ADDON_COMP_HSP_DOWNGRADE_ID: "Select a Hardware Support Package that " \
      "contains Components of matching or higher versions.",
   HSP_DOWNGRADE_ID: "Select a matching or higher version of Hardware " \
                     "Support Package.",
   HSP_COMP_DOWNGRADE_ID: "Select a Hardware Support Package that does not " \
      "downgrade an installed Hardware Support Package Component.",
   HSP_COMP_BASEIMAGE_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   HSP_COMP_ADDON_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   COMPONENT_DOWNGRADE_ID: "Replace the Component in the image with one of " \
      "the same or higher version.",
   SOLUTION_COMP_DOWNGRADE_ID: _COMMON_UPDATE_SOLUTION_COMP_RES,
   USER_COMP_BASEIMAGE_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   USER_COMP_ADDON_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   USER_COMP_HSP_DOWNGRADE_ID: _COMMON_ADD_USERCOMP_RES,
   USER_COMP_SOLUTION_DOWNGRADE_ID: _COMMON_UPDATE_SOLUTION_COMP_RES,
   VIB_DOWNGRADE_ID: "Add vendor Addon and Components that include matching " \
      "or higher version VIBs to the image.",

   ADDON_REMOVAL_ID: "Add the vendor Addon to the image to retain it.",
   COMPONENT_REMOVAL_ID: "Add the Components to the image to retain them.",
   HSP_REMOVAL_ID: "Enable the corresponding Hardware Support Manager to " \
                   "retain the package.",
   SOLUTIONCOMPONENT_REMOVAL_ID: "Enable corresponding functionalities of " \
      "the Solution to add the Components back to the image.",
   VIB_REMOVAL_ID: "To prevent them from being removed, include appropriate " \
      "components that are equivalent to these VIBs.",

   UNSUPPORTED_VENDOR_MODEL_ID % 'Vendor': "Replace vendor Addon or " \
      "Components that are released for a different hardware vendor.",
   UNSUPPORTED_VENDOR_MODEL_ID % 'Model': "Replace vendor Addon or " \
      "Components that are released for a different hardware model.",
   NO_NIC_DRIVER_ID: "Make sure the image contains the appropriate driver " \
      "for the management NIC.",
   NO_BOOTDEVICE_DRIVER_ID: "Make sure the image contains the appropriate " \
      "driver for the boot device.",
   INSUFFICIENT_BOOTBANK_SPACE_ID: "Remove unnecessary Components from " \
      "the image.",
   INSUFFICIENT_LOCKER_SPACE_ID: "Cleanup the locker partition on the host, " \
      "and retry check compliance.",
   DISK_TOO_SMALL_ID: "",
   HOST_ACCEPTANCE_ID: "Make sure the host is configured at an appropriate " \
      "acceptance level for the image, and retry check compliance.",
   VSAN_V1_DISK_ID: "Make sure all vSAN V1 disks have been upgraded to latest" \
      " disk format version.",

   COMPONENTS_REMOVED_DG: "Apply equal or higher versions of the " \
   "Component(s) on the host, and retry depot extraction.",
   ORPHAN_VIB: "Apply Component(s) that contain these VIB(s) for them to be " \
   "included in the extracted depot.",
   SEEDING_PENDING_REBOOT_ID: "Reboot the host to use the latest software change.",
   UNSUPPORTED_STATELESS_HOST_ID: "Retry getting software info and depot " \
      "extraction on a stateful host.",
   }

# Map from precheck result name to notification ID.
PRECHECK_NOTIFICATION_ID = {
   "MEMORY_SIZE": INSUFFICIENT_MEMORY_ID,
   "CPU_SUPPORT": UNSUPPORTED_CPU_ID,
   "CPU_CORES": NOT_ENOUGH_CPU_CORE_ID,
   "UNSUPPORTED_DEVICES": UNSUPPORTED_PCI_DEVICE,
   "VALIDATE_HOST_HW": UNSUPPORTED_VENDOR_MODEL_ID,
   "NATIVE_BOOT_NIC": NO_NIC_DRIVER_ID,
   "NATIVE_BOOTBANK": NO_BOOTDEVICE_DRIVER_ID,
   "IMAGEPROFILE_SIZE": INSUFFICIENT_BOOTBANK_SPACE_ID,
   "LOCKER_SPACE_AVAIL": INSUFFICIENT_LOCKER_SPACE_ID,
   "BOOT_DISK_SIZE": DISK_TOO_SMALL_ID,
   "HOST_ACCEPTANCE": HOST_ACCEPTANCE_ID,
   "VSAN_V1_DISK": VSAN_V1_DISK_ID,
   }

# Image entities.
BASE_IMAGE = "base_image"
ADDON = "addon"
COMPONENT = "component"
SOLUTION = "solution"
BASE_IMAGE_COMP = BASE_IMAGE + '_' + COMPONENT
ADDON_COMP = ADDON + '_' + COMPONENT
HARDWARE_SUPPORT_COMP = HARDWARE_SUPPORT + '_' + COMPONENT
SOLUTION_COMPONENT = SOLUTION + '_' + COMPONENT
OLD_ADDON_COMP = "old_addon_component"

# Convert component sources to image entities.
SOURCE_TO_ENTITY = {
   SOURCE_BASEIMAGE: BASE_IMAGE,
   SOURCE_ADDON: ADDON,
   SOURCE_HSP: HARDWARE_SUPPORT,
   SOURCE_SOLUTION: SOLUTION,
   SOURCE_USER: COMPONENT,
   }

# Map from downgrade entity to notification ID.
DOWNGRADE_NOTIFICATION_ID = {
   BASE_IMAGE: BASEIMAGE_DOWNGRADE_ID,
   ADDON: ADDON_DOWNGRADE_ID,
   COMPONENT: COMPONENT_DOWNGRADE_ID,
   HARDWARE_SUPPORT: HSP_DOWNGRADE_ID,
   HARDWARE_SUPPORT_COMP: HSP_COMP_DOWNGRADE_ID,
   }

# Component downgrade notification IDs indexed by from and to component source.
COMP_DOWNGRADE_NOTIFICATION_ID = {
   BASE_IMAGE: {
      BASE_IMAGE: BASEIMAGE_COMP_DOWNGRADE_ID,
      # BaseImage is always present so unless a component is removed from
      # BaseImage at one point, there should not be BaseImage to Addon/HSP
      # downgrades.
   },
   ADDON: {
      BASE_IMAGE: ADDON_COMP_BASEIMAGE_DOWNGRADE_ID,
      ADDON: ADDON_COMP_DOWNGRADE_ID,
      # When Addon is being removed, HSP might provide the same component.
      HARDWARE_SUPPORT: ADDON_COMP_HSP_DOWNGRADE_ID,
   },
   HARDWARE_SUPPORT: {
      BASE_IMAGE: HSP_COMP_BASEIMAGE_DOWNGRADE_ID,
      ADDON: HSP_COMP_ADDON_DOWNGRADE_ID,
      HARDWARE_SUPPORT: HSP_COMP_DOWNGRADE_ID,
   },
   SOLUTION: {
      # Solution components should not be found in another release unit.
      SOLUTION: SOLUTION_COMP_DOWNGRADE_ID,
   },
   COMPONENT: {
      # User component to release unit downgrades are reported by release
      # units.
      BASE_IMAGE: USER_COMP_BASEIMAGE_DOWNGRADE_ID,
      ADDON: USER_COMP_ADDON_DOWNGRADE_ID,
      HARDWARE_SUPPORT: USER_COMP_HSP_DOWNGRADE_ID,
      SOLUTION: USER_COMP_SOLUTION_DOWNGRADE_ID,
   }
}

# Image profile scan spec.
BASEIMAGE_COMPS_KEY = BASE_IMAGE_COMP + 's'
ADDON_COMPS_KEY = ADDON_COMP + 's'
# USER_COMP_KEY references all the components (Added, Upgraded or Downgraded) by
# the user
USER_COMPS_KEY = "user_components"
# USER_ADD_UPGRADE_COMPS_KEY references only the newly Added or Upgraded
# components  by the user
USER_ADD_UPGRADE_COMPS_KEY = "user_added_upgraded_components"
REMOVED_DG_BI_COMP_KEY = "removed_or_downgraded_bi_components"
REMOVED_DG_ADDON_COMP_KEY = "removed_or_downgraded_add_on_components"
ORPHAN_VIBS = 'orphan_vibs'

# Validate software spec entities.
VALIDATE_PREFIX = 'com.vmware.vcIntegrity.lifecycle.Validate.'
VALIDATE_SUCCESS_ID = VALIDATE_PREFIX + 'Success'
VALIDATE_SUCCESS_MSG = 'The image is valid.'

# Depot metadata schema validation entities
DEPOT_METADATA_SCHEMA_VALIDATE_SUCCESS_MSG = \
           'Depot validation succeeded.'
